' Towers of Hanoi for CMM2
' Rev 1.0.0 William M Leue 7/10/2020

option default integer

const NT = 3
const MAX_DISKS = 12

' constants for graphics
const HCORR = 1.0  ' change to 0.75 for 16:9 monitor with no letterboxing
const DISK_HEIGHT = 20
const MIN_DISK_WIDTH = 40
const DISK_WIDTH_INCR = 8
Const BASE_HEIGHT = 30
const PIN_WIDTH = 12
const BASE_COLOR = RGB(40, 15, 5)
const PIN_COLOR = RGB(40, 40, 40)
const CBLACK = RGB(BLACK)
const CNT_X = 400
const CNT_Y = 80

' Other global variables
dim speed = 0
dim num_disks = 0
dim num_moves = 0
dim last_tower_source = 0
dim last_disk_removed = 0
dim last_disk_number = 0
dim ss = 0
dim ms = 0

' Main Program
GetUserParameters
cls

' Variables defined only after num_disks has been set
dim Towers(NT, num_disks)
dim tptr(NT)
dim disk_colors(num_disks)

' Graphic Parameters
dim left_margin
dim top_margin
dim pin_height
dim tower_spacing
dim base_width
 
MakeGraphicParameters num_disks
NewGame num_disks
DrawTowers
DrawDisks

' Run an Automatic Solution
if ms = 1 then
  MoveTower num_disks, 0, 1, 2

' Run a Manual Solution
else
  UserMoveTower num_disks, 0, 1, 2
end if

end

' Ask the user questions that determines how the puzzle runs
sub GetUserParameters
  cls
  input "Want instructions? ", yn$
  if LEFT$(UCASE$(yn$), 1) = "Y" then
    PrintInstructions
  end if
  input "How Many Disks? (1..12): ", ud$
  num_disks = Val(ud$)
  if num_disks = 0 then
    exit
  end if
  input "Automatic solution (1) or manual (2)?: ", sms$
  ms = val(sms$)  
  if ms = 1 then
    input "Free run (1) or Single Step (2): ", ut$
    if val(ut$) = 2 then 
      ss = 1
      speed = 1
    else
      ss = 0
      input "What speed? (1..4=very fast, fast, med, slow): ", us$
      speed = VAL(us$)
    end if
  else 
    speed = 1
  end if
end sub

' Set up for a new Game
sub NewGame num
  local i
  for i = 0 to NT-1
    tptr(i) = -1
  next i
  for i = 0 to num-1
    Towers(0, i) = i+1
  next i
  tptr(0) = num-1
  MakeDiskColors num
  num_moves = 0
end sub

' Classic Recursive Solution
sub MoveTower disk, source, dest, spare
  if disk = 1 then
    MoveDisk source, dest)
  else
    MoveTower disk-1, source, spare, dest
    MoveDisk source, dest
    MoveTower disk-1, spare, dest, source
  END IF
END SUB

' Manual Solution
' Note: the delays after each key capture are
' Crucial to avoid key spillover!
' Note: this will work using either the regular
' number keys or the keypad keys and whether or not
' the Num Lock is active.
sub UserMoveTower disk, source, dest, spare
  local y, ust, udt, ok, key1, key2, done
  local state, nstate
  local xx$
  
  done = 0
  state = 0 : nstate = -1
  do
    if state = 0 then
      ok = 0
      ust = -1 : udt = -1
      do
        ok = 0
        key1 = 0
        if keydown(0) > 0 then
          key1 = keydown(1)
          pause 100
        end if
        ust = -1
        if key1 = 135 or key1 = 49 then ust = 1
        if key1 = 129 or key1 = 50 then ust = 2
        if key1 = 137 or key1 = 51 then ust = 3  
        if ust >= 1 and ust <= 3 then ok = 1
        if NumDisksOnTower(ust-1) = 0 then ok = 0
      loop until ok = 1
      nstate = 1
    else
      do
        ok = 0
        key2 = 0
        if keydown(0) > 0 then
          key2 = keydown(1)
          pause 100
        end if
        udt = -1
        if key2 = 135 or key2 = 49 then udt = 1
        if key2 = 129 or key2 = 50 then udt = 2
        if key2 = 137 or key2 = 51 then udt = 3  
        if udt >= 1 and udt <=3 then ok = 1
        if udt = ust then ok = 0
      loop until ust > 0 and udt > 0 and ok = 1
      MoveDisk ust-1, udt-1
      nstate = 0
    end if
    state = nstate
    if NumDisksOnTower(0) = 0 and NumDisksOnTower(1) = num_disks then
      done = 1
    end if
  loop until done = 1

  y = top_margin + pin_height + BASE_HEIGHT + 5
  if num_moves = 2^(num_disks)-1 then
    text 50, y, "Congratulations, You solved it in the minimum number of moves!"
  else
    text 50, y, "You solved it! (But it can be done in fewer moves.)"
  end if
end sub

' Move a single disk between towers
sub MoveDisk source, dest
  local tdisk
  
  if source < 0 or source > 2 or dest < 0 or dest > 2 then
    exit sub
  end if
  if source = dest then
    exit sub
  end if
  tdisk = Towers(source, tptr(source))
  Towers(source, tptr(source)) = 0
  last_tower_source = source
  last_disk_removed = tptr(source)
  last_disk_number = tdisk
  tptr(source) = tptr(source) - 1  
  tptr(dest) = tptr(dest) + 1
  Towers(dest, tptr(dest)) = tdisk
  num_moves = num_moves + 1
  DrawDisks
  if ss = 1 then
    do while keydown(0) = 0
      pause 1
    loop
    do while keydown(0) > 0
      pause 1
    loop 
  else
    pause 20 + (speed-1)*200
  end if
END SUB

' returns the number of disks on the specified tower
function NumDisksOnTower(tower)
  if tower < 0 or tower > 2 then
    NumDisksOnTower = 0
    exit function
  end if
  NumDisksOnTower = tptr(tower) + 1
end function

' uses the HSV to RGB conversion subroutine to
' make a rainbow of colors for the disks
sub MakeDiskColors num  
  local i, s, v, r, g, b
  local float hue, ainc
  ainc = 360.0/num
  for i = 0 to num-1
    hue = ainc*i
    HSV2RGB hue, 1.0, 1.0, r, g, b
    disk_colors(i) = RGB(r, g, b)
  next i    
end sub

' Compute graphic parameters from number of disks
sub MakeGraphicParameters num
  local xc = 400
  local biggest_disk_width = MIN_DISK_WIDTH + 2*(num-1)*DISK_WIDTH_INCR  
  base_width = biggest_disk_width + 20
  tower_spacing = base_width + 20
  left_margin = xc - HCORR*base_width - 20
  pin_height = num*DISK_HEIGHT + 20
  top_margin = (600 - pin_height - BASE_HEIGHT)/2
end sub

' Graphical Rendering of Towers
sub DrawTowers
  local i, j, xc, xp, x1, y1, x2, y2, w
  local xv(11), yv(11)
  local last_tower, last_disk

  ' render the towers
  for i = 0 to NT-1
    xc = left_margin + HCORR*i*tower_spacing
    x1 = xc - HCORR*base_width/2
    y1 = top_margin + pin_height
    x2 = x1 + HCORR*base_width
    line x1, y1, x2, y1, BASE_HEIGHT, BASE_COLOR
    xp = xc - HCORR*PIN_WIDTH/2
    y1 = top_margin
    y2 = top_margin + pin_height
    line xp, y1, xp, y2, HCORR*PIN_WIDTH, PIN_COLOR  
    circle xc, y1, HCORR*PIN_WIDTH/2, 4, HCORR, PIN_COLOR, PIN_COLOR
    text xc, y2+BASE_HEIGHT/2-4, str$(i+1)
  next i

end sub

' render the disks
sub DrawDisks
  local i, j, h, xc, x1, y1, x2, y2, c, nd, xoff
  local xv(MAX_DISKS), yv(MAX_DISKS)

  for i = 0 to NT-1
    xc = left_margin + HCORR*i*tower_spacing
    xp = xc - HCORR*PIN_WIDTH/2

    ' render the disks
    for j = 0 to tptr(i)
      h = DISK_HEIGHT
      y1 = top_margin + pin_height - (j+1)*DISK_HEIGHT
      y2 = y1 + h
      d = Towers(i, j)
      x1 = xc - HCORR*MIN_DISK_WIDTH/2 - HCORR*(num_disks-d)*DISK_WIDTH_INCR
      w = HCORR*MIN_DISK_WIDTH + HCORR*(num_disks-d)*DISK_WIDTH_INCR*2
      xv(0) = x1         : yv(0) = y1
      xv(1) = x1+w       : yv(1) = y1
      xv(2) = x1+w+.2*h  : yv(2) = y1 + .25*h
      xv(3) = x1+w+.22*h : yv(3) = y1 + .5*h
      xv(4) = x1+w+.2*h  : yv(4) = y1 + .75*h
      xv(5) = x1+w       : yv(5) = y2
      xv(6) = x1         : yv(6) = y2
      xv(7) = x1-.2*h    : yv(7) = y2 - .25*h
      xv(8) = x1-.22*h   : yv(8) = y2 - .5*h
      xv(9) = x1-.2*h    : yv(9) = y2 - .75*h
      xv(10) = xv(0)     : yv(10) = yv(0)
      c = disk_colors(num_disks-d)
      polygon 11, xv(), yv(), CBLACK, c)       
    next j
 
  next i

  ' erase the disk and repair the tower pin where a disk was removed from a tower
  if num_moves > 0 then
    xc = left_margin + HCORR*last_tower_source*tower_spacing
    xp = xc - HCORR*PIN_WIDTH/2
    x1 = xc - HCORR*MIN_DISK_WIDTH/2 - HCORR*(num_disks-last_disk_number)*DISK_WIDTH_INCR
    y1 = top_margin + pin_height - (last_disk_removed+1)*h
    w = HCORR*MIN_DISK_WIDTH + HCORR*(num_disks-last_disk_number)*DISK_WIDTH_INCR*2
    y2 = y1 + h
    xv(0) = x1         : yv(0) = y1
    xv(1) = x1+w       : yv(1) = y1
    xv(2) = x1+w+.2*h  : yv(2) = y1+.25*h
    xv(3) = x1+w+.22*h : yv(3) = y1+.5*h
    xv(4) = x1+w+.2*h  : yv(4) = y1+.75*h
    xv(5) = x1+w       : yv(5) = y2
    xv(6) = x1         : yv(6) = y2
    xv(7) = x1-.2*h    : yv(7) = y2-.25*h
    xv(8) = x1-.22*h   : yv(8) = y2-.5*h
    xv(9) = x1-.2*h    : yv(9) = y2-.75*h
    xv(10) = xv(0)     : yv(10) = yv(0)
    polygon 11, xv(), yv(), CBLACK, CBLACK
    line xp, y1, xp, y2, HCORR*PIN_WIDTH, PIN_COLOR
  end if

  ' draw the move count
  nd = 1
  if num_moves > 9 then nd = 2
  if num_moves > 99 then nd = 3
  if num_moves > 999 then nd = 4
  xoff = nd*24 + 5
  box CNT_X - 50, top_MARGIN-CNT_Y, 200, 30,, CBLACK, CBLACK 
  text CNT_X-xoff, top_margin - CNT_Y, str$(num_moves),, 1, 5

end sub

' Convert HSV colors to RGB
sub HSV2RGB h,s,v,r,g,b
  local float hh!, f!, p!, q!, t!
  local float rp!, gp!, bp!
  hh! = h/60.0
  i = INT(hh)
  f! = hh! - i
  p! = v*(1.0-s)
  q! = v*(1.0-s*f!)
  t! = v*(1.0-s*(1-f!))

  select case i
    case 0
      rp! = v : gp! = t! : bp! = p!
    case 1
      rp! = q! : gp! = v : bp! = p!
    case 2
      rp! = p! : gp! = v : bp! = t!
    case 3
      rp! = p! : gp! = q! : bp! = v
    case 4
      rp! = t! : gp! = p! : bp! = v
    case 5
      rp! = v : gp! = p! : bp! = q!
  end select
  r = rp!*255 : g = gp!*255 : b = bp!*255
end sub

' clues for the clueless
sub PrintInstructions
  cls
  print "The object of the Towers of Hanoi puzzle is to move the stack of colored disks"
  print "from the left-hand tower to the center tower. You only move one disk at a time,"
  print "and you can NEVER have a larger disk on top of a smaller disk."
  print ""
  print "The optimum solution requires 2 raised to the power of the number of disks, "
  print "minus one. For instance, if there are 5 disks, 31 moves are required."
  print ""
  print "You can choose to watch the computer solve the puzzle, or you can solve it yourself."
  print "You can choose how many disks will be in the puzzle, from 1 to 12 disks."
  print "If you choose an automatic solution, you can choose the speed that disks are moved."
  print "You can also choose to single-step the automatic solution."
  print ""
  print "If you choose a manual solution, you use the number keys 1, 2, and 3 to select the"
  print "towers. Your first entry chooses the source tower for a disk move, and the second"
  print "entry chooses the destination tower for that disk."
  print "If you solve the puzzle, the computer will tell you if you matched the best possible"
  print "solution.
  print ""
  print "Hint: start with just one disk, then, two, then three, and so on to get insight"
  print "into the pattern of disk moves that lead to the best solution."
  print ""
  print "In the legendary version of this puzzle, the tower has 64 disks. This number of disks"
  print "would require 2 to the 64th power, minus one moves. That is about 1.8 followed by
  print "nineteen zeros, or 1.8 quintillion! That would take a while to complete....:"
  print ""
  print "If you are mathematically inclined, you may already know that the Towers of Hanoi"
  print "is isomorphic to several other puzzles, including the Chinese Ring puzzle."
  print ""
  xx$ = INKEY$
  print ""
  print "Press any key to continue..."
  do
  loop until INKEY$ <> ""
  cls
end sub
